package furny.swing.admin.statistics;

import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.beans.PropertyVetoException;
import java.util.ArrayList;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.Box;
import javax.swing.DefaultListModel;
import javax.swing.JButton;
import javax.swing.JDesktopPane;
import javax.swing.JFrame;
import javax.swing.JInternalFrame;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.ListSelectionModel;
import javax.swing.SwingWorker;
import javax.swing.UIManager;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import org.jdesktop.swingx.JXBusyLabel;
import org.jdesktop.swingx.JXLabel;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;

import furny.ga.logger.StatisticsDBManager;
import furny.swing.admin.statistics.chart.FitnessHistogramChart;
import furny.swing.admin.statistics.chart.FurnitureHistogramChart;
import furny.swing.admin.statistics.chart.IChartSettings;
import furny.swing.admin.statistics.chart.MaxFitnessGenerationChart;
import furny.swing.admin.statistics.chart.MaxFitnessTimeChart;
import furny.swing.admin.statistics.chart.MeanFitnessGenerationChart;
import furny.swing.admin.statistics.chart.MeanFitnessTimeChart;
import furny.swing.admin.statistics.chart.MinMaxFitnessGenerationChart;

/**
 * Frame for statistics and analytics. Displays charts of former evolutionary
 * runs.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
@SuppressWarnings("serial")
public class StatisticsFrame extends JFrame {
  private final JList chartList;
  private final ChartListModel chartModel;

  private final JList runList;
  private final RunListModel runModel;
  private JDesktopPane desktop;

  private JPanel rightPanel;

  private Long runId;
  private IChartSettings settings;

  /**
   * Instantiates a new statistics frame.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public StatisticsFrame() {
    super("Statistics DB");
    setDefaultCloseOperation(EXIT_ON_CLOSE);

    setLayout(new GridBagLayout());
    final GridBagConstraints constraints = new GridBagConstraints();
    constraints.gridx = 0;
    constraints.gridy = 0;
    constraints.insets = new Insets(5, 5, 5, 5);
    constraints.fill = GridBagConstraints.BOTH;
    constraints.weightx = 1.0d;
    constraints.weighty = 1.0d;

    chartModel = new ChartListModel();
    chartList = new JList(chartModel);
    chartList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    chartList.addListSelectionListener(new ListSelectionListener() {

      @Override
      public void valueChanged(final ListSelectionEvent e) {
        if (!e.getValueIsAdjusting()) {
          settings = (IChartSettings) chartModel.getElementAt(chartList
              .getSelectedIndex());

          openChartFrame();
        }
      }
    });
    runModel = new RunListModel();
    runList = new JList(runModel);
    runList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    runList.addListSelectionListener(new ListSelectionListener() {

      @Override
      public void valueChanged(final ListSelectionEvent e) {
        if (!e.getValueIsAdjusting()) {
          runId = (Long) runModel.getElementAt(runList.getSelectedIndex());

          openChartFrame();
        }
      }
    });

    JSplitPane splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT,
        new JScrollPane(chartList), new JScrollPane(runList));

    splitPane.setDividerLocation(300);

    UIManager.put("Desktop.background", chartList.getBackground());

    createRightPanel();

    splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, splitPane,
        rightPanel);

    splitPane.setDividerLocation(200);

    add(splitPane, constraints);

    pack();
    setSize(800, 600);
    setLocationRelativeTo(null);
  }

  /**
   * Creates the right panel.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void createRightPanel() {
    rightPanel = new JPanel();

    rightPanel.setLayout(new GridBagLayout());
    final GridBagConstraints constraints = new GridBagConstraints();
    constraints.gridx = 0;
    constraints.gridy = 0;
    constraints.fill = GridBagConstraints.NONE;
    constraints.weightx = 0.0d;
    constraints.weighty = 0.0d;

    rightPanel.add(new JButton(new ActionTileViews()), constraints);

    constraints.gridx++;

    rightPanel.add(new JButton(new ActionCascadeViews()), constraints);

    constraints.gridx++;

    rightPanel.add(new JButton(new ActionCloseAllViews()), constraints);

    constraints.gridx++;
    rightPanel.add(Box.createHorizontalGlue(), constraints);

    constraints.gridx = 0;
    constraints.gridy++;
    constraints.gridwidth = GridBagConstraints.REMAINDER;
    constraints.weightx = 1.0d;
    constraints.weighty = 1.0d;
    constraints.fill = GridBagConstraints.BOTH;
    desktop = new JDesktopPane();

    rightPanel.add(desktop, constraints);
  }

  /**
   * Opens the chart frame.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void openChartFrame() {
    if (runId != null && settings != null) {

      final String title = settings.toString() + " evaluation run " + runId;

      for (final JInternalFrame frame : desktop.getAllFrames()) {
        // if the frame is already opened, move it to the front
        if (frame.getTitle().equals(title) && frame.isVisible()) {
          try {
            frame.setIcon(false);
          } catch (final PropertyVetoException e) {
            e.printStackTrace();
          }
          frame.moveToFront();

          return;
        }
      }

      final JInternalFrame frame = new JInternalFrame(title, true, true, true,
          true);

      final JXBusyLabel busyLabel = new JXBusyLabel(new Dimension(50, 50));
      busyLabel.setHorizontalAlignment(JXLabel.CENTER);
      busyLabel.setBusy(true);

      frame.add(busyLabel);
      frame.setBounds(desktop.getComponentCount() * 25,
          desktop.getComponentCount() * 20, 400, 300);

      desktop.add(frame);

      final SwingWorker<Void, Void> sw = new SwingWorker<Void, Void>() {
        private JFreeChart chart;

        @Override
        protected Void doInBackground() throws Exception {
          try {
            chart = settings.createChart(runId);
          } catch (final Throwable t) {
            t.printStackTrace();
          }

          return null;
        }

        @Override
        protected void done() {
          final ChartPanel chartPanel = new ChartPanel(chart);
          frame.remove(busyLabel);

          frame.add(chartPanel);
          frame.setSize(frame.getSize());
        }
      };
      sw.execute();

      frame.setVisible(true);
    }
  }

  /**
   * List model displaying available charts.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ChartListModel extends DefaultListModel {
    private final List<IChartSettings> settings = new ArrayList<IChartSettings>();

    /**
     * Instantiates a new chart list model.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ChartListModel() {
      settings.add(new MinMaxFitnessGenerationChart());
      settings.add(new MaxFitnessGenerationChart());
      settings.add(new MaxFitnessTimeChart());
      settings.add(new MeanFitnessGenerationChart());
      settings.add(new MeanFitnessTimeChart());
      settings.add(new FitnessHistogramChart());
      settings.add(new FurnitureHistogramChart());
    }

    @Override
    public Object getElementAt(final int index) {
      return settings.get(index);
    }

    @Override
    public int getSize() {
      return settings.size();
    }

  }

  /**
   * List model displaying former runs.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class RunListModel extends DefaultListModel {
    private final List<Long> runIds;

    /**
     * Instantiates a new run list model.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public RunListModel() {
      runIds = StatisticsDBManager.getInstance().getRunEntries();
    }

    @Override
    public Object getElementAt(final int index) {
      return runIds.get(index);
    }

    @Override
    public int getSize() {
      return runIds.size();
    }

  }

  /**
   * Action to arrange frames in a tile pattern.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionTileViews extends AbstractAction {

    /**
     * Instantiates a new action to tile views.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionTileViews() {
      super("Grid");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      final Dimension d = desktop.getSize();
      final JInternalFrame[] frames = desktop.getAllFrames();
      final int n = frames.length;
      if (n == 0) {
        return;
      }
      final int nx = (int) Math.sqrt(n);
      int ny = (int) (Math.ceil(((double) frames.length) / nx));
      final int ymax = frames.length - nx * (ny - 1);
      int w, h;

      if (ymax == 0) {
        ny--;
        h = d.height / ny;
      } else {
        h = d.height / ny;
        if (ymax < ny) {
          ny--;
          w = d.width / ymax;
          for (int x = 0; x < ymax; x++) {
            frames[nx * ny + x].setBounds(x * w, ny * h, w, h);

            // demaximize and deiconify all frames
            try {
              frames[nx * ny + x].setMaximum(false);
              frames[nx * ny + x].setIcon(false);
            } catch (final PropertyVetoException e1) {
              e1.printStackTrace();
            }
          }
        }
      }

      w = d.width / nx;
      for (int y = 0; y < ny; y++) {
        for (int x = 0; x < nx; x++) {
          frames[x + y * nx].setBounds(x * w, y * h, w, h);

          // demaximize and deiconify all frames
          try {
            frames[x + y * nx].setMaximum(false);
            frames[x + y * nx].setIcon(false);
          } catch (final PropertyVetoException e1) {
            e1.printStackTrace();
          }
        }
      }
    }
  }

  /**
   * Action to arrange frames in a cascade pattern.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionCascadeViews extends AbstractAction {

    /**
     * Instantiates a new action to cascade views.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionCascadeViews() {
      super("Cascade");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      final Dimension d = desktop.getSize();
      final JInternalFrame[] frames = desktop.getAllFrames();
      final int n = frames.length;
      if (n == 0) {
        return;
      }
      final int s = frames[0].getRootPane().getLocation().y;
      final int m = n * s;
      final int w = d.width - m;
      final int h = d.height - m;

      for (int i = 0; i < n; i++) {
        // demaximize and deiconify all frames
        try {
          frames[n - i - 1].setMaximum(false);
          frames[n - i - 1].setIcon(false);
        } catch (final PropertyVetoException e1) {
          e1.printStackTrace();
        }

        frames[n - i - 1].setBounds(i * s, i * s, w + s, h + s);
      }
    }
  }

  /**
   * Action to close all views.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionCloseAllViews extends AbstractAction {

    /**
     * Instantiates a new action to close all views.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionCloseAllViews() {
      super("Close all");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      final JInternalFrame[] frames = desktop.getAllFrames();
      for (final JInternalFrame frame : frames) {
        try {
          frame.setClosed(true);
        } catch (final PropertyVetoException e1) {
          e1.printStackTrace();
        }
      }
    }
  }
}
